package net.argius.stew.gui;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.sql.*;
import java.text.*;

import javax.swing.*;
import javax.swing.event.*;

import net.argius.logging.*;
import net.argius.stew.*;

/**
 * GUIEBhȄo͏B
 */
final class WindowOutputProcessor implements OutputProcessor {

    private static final Logger log = LoggerFactory.getLogger(WindowOutputProcessor.class);

    Window window;
    JSplitPane splitPane;
    ResultSetTable resultSetTable;
    ConsoleTextArea consoleTextArea;
    JLabel statusBarLabel;

    /*
     * refresh邽߂̎d|
     */
    final ThreadLocal lastCommandInfo;

    private Environment env;
    private WindowLauncher launcher;
    private TextSearchPanel textSearchPanel;
    private FontChooser fontChooser;
    private ThreadLocal currentCommand;

    /**
     * WindowOutputProcessor̐B
     * @param env
     * @param launcher
     */
    WindowOutputProcessor(Environment env, WindowLauncher launcher) {
        this.lastCommandInfo = new ThreadLocal();
        try {
            this.env = env;
            this.launcher = launcher;
            this.fontChooser = new FontChooser();
            this.currentCommand = new ThreadLocal();
            env.setOutputProcessor(this);
            initializeComponents();
            fontChooser.addChangeListener(new ChangeListener() {

                public void stateChanged(ChangeEvent e) {
                    Object source = e.getSource();
                    if (source instanceof Font) {
                        onFontChanged((Font)source);
                    }
                }

            });
            // \
            window.loadConfiguration();
            window.setVisible(true);
            output(new Prompt(env));
            consoleTextArea.requestFocus();
        } catch (RuntimeException ex) {
            onError(ex);
            close();
        }
    }

    /**
     * eR|[lg̏B
     */
    private void initializeComponents() {
        window = new Window(env, this);
        window.addWindowListener(new WindowAdapter() {

            public void windowClosing(WindowEvent e) {
                onWindowClosing();
            }

        });
        // table area
        resultSetTable = new ResultSetTable(env);
        resultSetTable.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
        resultSetTable.setColumnSelectionAllowed(true);
        JScrollPane resultPane = new JScrollPane(resultSetTable);
        resultPane.setVisible(true);
        // message area
        consoleTextArea = new ConsoleTextArea(this);
        consoleTextArea.setMargin(new Insets(4, 8, 4, 4));
        consoleTextArea.addMouseListener(new ContextMenu(window));
        JScrollPane messagePane = new JScrollPane(consoleTextArea);
        messagePane.setHorizontalScrollBarPolicy(ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER);
        messagePane.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);
        textSearchPanel = new TextSearchPanel(window, new TextSearch[]{resultSetTable,
                                                                       consoleTextArea});
        // status bar
        JPanel statusBar = new JPanel(new BorderLayout());
        statusBarLabel = new JLabel(" ");
        statusBarLabel.setVisible(false);
        statusBar.add(statusBarLabel, BorderLayout.LINE_START);
        statusBarLabel.setFont(FormedFont.Small);
        statusBarLabel.setForeground(Color.BLUE);
        // menu
        Menu menu = new Menu(env, this);
        // layout
        final JPanel p = new JPanel(new BorderLayout());
        p.add(messagePane, BorderLayout.CENTER);
        p.add(textSearchPanel, BorderLayout.SOUTH);
        splitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
        splitPane.setTopComponent(resultPane);
        splitPane.setBottomComponent(p);
        splitPane.setResizeWeight(0.3);
        splitPane.validate();
        window.setJMenuBar(menu);
        window.getContentPane().add(splitPane, BorderLayout.CENTER);
        window.getContentPane().add(statusBar, BorderLayout.PAGE_END);
    }

    /* (overridden)
     * @see net.argius.stew.OutputProcessor#output(java.lang.Object)
     */
    public void output(Object object) {
        try {
            if (object instanceof ResultSetReference) {
                ResultSetReference ref = (ResultSetReference)object;
                outputResultSetReference(ref);
                final Object command = currentCommand.get();
                if (command != null && command instanceof String) {
                    String name = String.valueOf(env.getCurrentConnection());
                    final NamedValue info = new NamedValue(name, command);
                    EventQueue.invokeLater(new Runnable() {

                        public void run() {
                            lastCommandInfo.set(info);
                        }

                    });
                }
            } else if (object instanceof Prompt) {
                consoleTextArea.append(object.toString());
            } else {
                StringWriter buffer = new StringWriter();
                PrintWriter out = new PrintWriter(buffer);
                out.println(object);
                consoleTextArea.append(buffer.toString());
            }
            consoleTextArea.validate();
        } catch (RuntimeException ex) {
            onError(ex);
        }
    }

    /* (overridden)
     * @see net.argius.stew.OutputProcessor#close()
     */
    public void close() {
        textSearchPanel.setVisible(false);
        window.validate();
        window.dispose();
        launcher.remove(this);
        env.release();
    }

    /**
     * \ŐV̏ԂɂB
     */
    void refresh() {
        Connection conn = env.getCurrentConnection();
        if (conn == null) {
            return;
        }
        Object o = lastCommandInfo.get();
        if (o != null && o instanceof NamedValue) {
            NamedValue info = (NamedValue)o;
            if (conn.toString().equals(info.getName())) {
                consoleTextArea.replace((String)info.getValue());
                consoleTextArea.sendCommand();
            }
        }
    }

    /**
     * ResultSetQƂo͂B
     * @param ref ResultSetQ
     */
    private void outputResultSetReference(ResultSetReference ref) {
        try {
            ColumnOrder order = ref.getOrder();
            String command = (String)currentCommand.get();
            int count = resultSetTable.outputResultSet(ref.getResultSet(),
                                                       order,
                                                       command);
            ref.setRecordCount(count);
        } catch (SQLException ex) {
            log.error("", ex);
            onError(ex);
        }
    }

    /**
     * AԂ̎擾B
     * @return A
     */
    int getSequencialNumber() {
        return launcher.getSequentialNumber();
    }

    /**
     * R}hs˗B
     * @param commandString R}h
     */
    void requestCommand(String commandString) {
        requestCommand(commandString, StringClass.EMPTY);
    }

    /**
     * R}hs˗B
     * @param commandString R}h
     * @param message Mɕ\郁bZ[W
     */
    void requestCommand(String commandString, String message) {
        final Environment env = this.env;
        final ThreadLocal currentCommand = this.currentCommand;
        final String command = commandString;
        output(message);
        window.setAllComponentsEnabled(false);
        Thread thread = new Thread() {

            public void run() {
                try {
                    currentCommand.set(command);
                    final double execTime = executeCommand(command);
                    EventQueue.invokeLater(new Runnable() {

                        public void run() {
                            window.setAllComponentsEnabled(true);
                            output(new Prompt(env));
                            setStatusMessage(command, execTime);
                            consoleTextArea.requestFocus();
                        }

                    });
                } catch (Throwable th) {
                    onError(th);
                    window.setAllComponentsEnabled(true);
                }
            }

            /**
             * R}hsB
             * @param command R}h
             * @return s
             */
            private double executeCommand(String command) {
                long startTime = System.currentTimeMillis();
                boolean result = Command.invoke(env, command);
                if (!result) {
                    close();
                }
                return (System.currentTimeMillis() - startTime) / 1000d;
            }

        };
        thread.setDaemon(true);
        thread.start();
    }

    /**
     * Xe[^XbZ[W̐ݒB
     * @param command R}h
     * @param requiredTime s
     */
    void setStatusMessage(String command, double requiredTime) {
        String formatString = Messages.getString("window.StatusBar.format");
        Object[] arguments = {command, new Double(requiredTime)};
        String message = MessageFormat.format(formatString, arguments);
        statusBarLabel.setText(message);
    }

    /**
     * ̓GAɈړB
     */
    void gotoInput() {
        consoleTextArea.requestFocus();
    }

    /**
     * ڑݒ_CAO̕\˗B
     */
    void requestShowConnectorMapEditDialog() {
        ConnectorMapEditDialog dialog = new ConnectorMapEditDialog(window, env);
        dialog.pack();
        dialog.setModal(true);
        dialog.setLocationRelativeTo(window);
        dialog.setVisible(true);
    }

    /* (overridden)
     * @see java.lang.Object#finalize()
     */
    protected void finalize() throws Throwable {
        log.info("removing : " + this);
        super.finalize();
    }

    /**
     * tHg̎擾B
     * @return tHg
     */
    Font getFont() {
        return fontChooser.getCurrentFont();
    }

    /**
     * tHgύX̏B
     * @param newFont ύX̃tHg
     */
    void onFontChanged(Font newFont) {
        fontChooser.setCurrentFont(newFont);
        resultSetTable.setFont(newFont);
        consoleTextArea.setFont(newFont);
    }

    /**
     * VEBhEJB
     */
    void createNew() {
        launcher.launch(new Environment(env));
    }

    /**
     * vȌIvB
     */
    void requestExit() {
        launcher.closeAll();
    }

    /**
     * TextSearchPanel̕\vB
     * @param target Ώ
     */
    void requestShowTextSearchPanel(TextSearch target) {
        textSearchPanel.setTarget(target);
        textSearchPanel.setVisible(true);
    }

    /**
     * FontChooser̕\vB
     */
    void requestShowFontChooser() {
        fontChooser.showDialog(window);
    }

    /**
     * EBhE̏B
     */
    void onWindowClosing() {
        int windowCount = launcher.getWindowCount();
        if (windowCount <= 1) {
            Menu menu = (Menu)window.getJMenuBar();
            menu.onMenuQuitSelected();
        } else {
            Connection conn = env.getCurrentConnection();
            boolean connecting;
            try {
                connecting = (conn != null && !conn.isClosed());
            } catch (SQLException ex) {
                onError(ex);
                return;
            }
            if (connecting) {
                String message = Messages.getString("menu.Menu.message.closewindow");
                if (DialogMessage.confirmYesNo(window, message) != JOptionPane.YES_OPTION) {
                    return;
                }
            }
            close();
        }
    }

    /**
     * G[B
     * @param th G[
     */
    void onError(Throwable th) {
        log.error("", th);
        String message = th.getMessage();
        if (StringClass.isEmpty(message)) {
            message = th.toString();
        }
        DialogMessage.alert(window, message);
    }

}