package net.argius.stew.gui;

import java.awt.*;
import java.awt.datatransfer.*;
import java.awt.event.*;
import java.io.*;
import java.sql.*;
import java.util.*;
import java.util.List;

import javax.swing.*;
import javax.swing.table.*;

import net.argius.logging.*;
import net.argius.stew.io.*;

/**
 * ResultSetTableActionB
 */
final class ResultSetTableAction {

    /**
     * <code>JUMP_TO_HOME</code>
     */
    static final String JUMP_TO_HOME = "jump-to-home";

    /**
     * <code>JUMP_TO_END</code>
     */
    static final String JUMP_TO_END = "jump-to-end";

    /**
     * <code>JUMP_TO_TOP</code>
     */
    static final String JUMP_TO_TOP = "jump-to-top";

    /**
     * <code>JUMP_TO_BOTTOM</code>
     */
    static final String JUMP_TO_BOTTOM = "jump-to-bottom";

    /**
     * <code>JUMP_TO_LEFTMOST</code>
     */
    static final String JUMP_TO_LEFTMOST = "jump-to-leftmost";

    /**
     * <code>JUMP_TO_RIGHTMOST</code>
     */
    static final String JUMP_TO_RIGHTMOST = "jump-to-rightmost";

    /**
     * <code>SELECT_TO_HOME</code>
     */
    static final String SELECT_TO_HOME = "select-to-home";

    /**
     * <code>SELECT_TO_END</code>
     */
    static final String SELECT_TO_END = "select-to-end";

    /**
     * <code>SELECT_TO_TOP</code>
     */
    static final String SELECT_TO_TOP = "select-to-top";

    /**
     * <code>SELECT_TO_BOTTOM</code>
     */
    static final String SELECT_TO_BOTTOM = "select-to-bottom";

    /**
     * <code>SELECT_TO_LEFTMOST</code>
     */
    static final String SELECT_TO_LEFTMOST = "select-to-leftmost";

    /**
     * <code>SELECT_TO_RIGHTMOST</code>
     */
    static final String SELECT_TO_RIGHTMOST = "select-to-rightmost";

    /**
     * <code>ADJUST_COLUMN_SIZE</code>
     */
    static final String ADJUST_COLUMN_SIZE = "adjust-column-size";

    /**
     * <code>COPY_CELL_WITH_ESCAPE</code>
     */
    static final String COPY_CELL_WITH_ESCAPE = "copy-cell-with-escape";

    /**
     * <code>COPY_COLUMN_NAMES</code>
     */
    static final String COPY_COLUMN_NAMES = "copy-column-names";

    /**
     * <code>PASTE_TO_SELECTED_CELLS</code>
     */
    static final String PASTE_TO_SELECTED_CELLS = "paste-to-selectedCells";

    /**
     * <code>SET_NULL_TO_SELECTED_CELLS</code>
     */
    static final String SET_NULL_TO_SELECTED_CELLS = "set-null-to-selectedCells";

    /**
     * <code>ADD_NEW_ROW</code>
     */
    static final String ADD_NEW_ROW = "add-new-row";

    /**
     * <code>ADD_DUPLICATE_ROW</code>
     */
    static final String ADD_DUPLICATE_ROW = "add-duplicate-row";

    /**
     * <code>INSERT_FROM_CLIPBOARD</code>
     */
    static final String INSERT_FROM_CLIPBOARD = "insert-from-clipboard";

    /**
     * <code>INSERT_ADDED_ROWS</code>
     */
    static final String INSERT_ADDED_ROWS = "insert-added-rows";

    /**
     * <code>DELETE_SELECTED_ROWS</code>
     */
    static final String DELETE_SELECTED_ROWS = "delete-selected-rows";

    private static final Logger log = LoggerFactory.getLogger(ResultSetTableAction.class);

    final ResultSetTable table;
    final ContextMenu contextMenu;

    private final InputMap imap;
    private final ActionMap amap;

    /**
     * ResultSetTableAction̐B
     * @param table ResultSetTable 
     */
    private ResultSetTableAction(ResultSetTable table) {
        this.table = table;
        this.contextMenu = new ContextMenu(table);
        this.imap = table.getInputMap();
        this.amap = table.getActionMap();
    }

    /**
     * Action蓖ĂB
     * @param table ResultSetTable
     */
    static void bindAction(final ResultSetTable table) {
        ResultSetTableAction a = new ResultSetTableAction(table);
        a.bindJumpTo(JUMP_TO_HOME, KeyEvent.VK_HOME, false);
        a.bindJumpTo(JUMP_TO_END, KeyEvent.VK_END, false);
        a.bindJumpTo(JUMP_TO_TOP, KeyEvent.VK_UP, false);
        a.bindJumpTo(JUMP_TO_BOTTOM, KeyEvent.VK_DOWN, false);
        a.bindJumpTo(JUMP_TO_LEFTMOST, KeyEvent.VK_LEFT, false);
        a.bindJumpTo(JUMP_TO_RIGHTMOST, KeyEvent.VK_RIGHT, false);
        a.bindJumpTo(SELECT_TO_HOME, KeyEvent.VK_HOME, true);
        a.bindJumpTo(SELECT_TO_END, KeyEvent.VK_END, true);
        a.bindJumpTo(SELECT_TO_TOP, KeyEvent.VK_UP, true);
        a.bindJumpTo(SELECT_TO_BOTTOM, KeyEvent.VK_DOWN, true);
        a.bindJumpTo(SELECT_TO_LEFTMOST, KeyEvent.VK_LEFT, true);
        a.bindJumpTo(SELECT_TO_RIGHTMOST, KeyEvent.VK_RIGHT, true);
        a.bindAdjustColumnSize();
        a.bindCopyCellWithEscape();
        a.bindCopyColumnNames();
        a.bindPasteToSelectedCells();
        a.bindSetNullToSelectedCells();
        a.bindAddNewRow();
        a.bindAddDuplicateRow();
        a.bindInsertFromClipboard();
        a.bindInsertAddedRows();
        a.bindDeleteSelectedRows();
    }

    /**
     * JUMP_TO_̊蓖āB
     * @param name ANV
     * @param key ăL[ 
     * @param withSelect I[h
     */
    private void bindJumpTo(String name, int key, boolean withSelect) {
        final boolean extend = withSelect;
        final CellCursor c = new CellCursor(table);
        final Action action;
        if (name.endsWith("-to-home")) {
            action = new AbstractAction() {

                public void actionPerformed(ActionEvent e) {
                    table.changeSelection(0, 0, false, extend);
                }

            };
        } else if (name.endsWith("-to-end")) {
            action = new AbstractAction() {

                public void actionPerformed(ActionEvent e) {
                    int rowCount = table.getRowCount();
                    int columnCount = table.getColumnCount();
                    table.changeSelection(rowCount - 1, columnCount - 1, false, extend);
                }

            };
        } else if (name.endsWith("-to-top")) {
            action = new AbstractAction() {

                public void actionPerformed(ActionEvent e) {
                    table.changeSelection(0, c.getColumnPosition(), false, extend);
                }

            };
        } else if (name.endsWith("-to-bottom")) {
            action = new AbstractAction() {

                public void actionPerformed(ActionEvent e) {
                    int rowCount = table.getRowCount();
                    table.changeSelection(rowCount - 1, c.getColumnPosition(), false, extend);
                }

            };
        } else if (name.endsWith("-to-leftmost")) {
            action = new AbstractAction() {

                public void actionPerformed(ActionEvent e) {
                    table.changeSelection(c.getRowPosition(), 0, false, extend);
                }

            };
        } else if (name.endsWith("-to-rightmost")) {
            action = new AbstractAction() {

                public void actionPerformed(ActionEvent e) {
                    int columnCount = table.getColumnCount();
                    table.changeSelection(c.getRowPosition(), columnCount - 1, false, extend);
                }

            };
        } else {
            log.warn("unknown key: " + name);
            return;
        }
        amap.put(name, action);
        final int modifiers = InputEvent.CTRL_DOWN_MASK
                              | (withSelect ? InputEvent.SHIFT_DOWN_MASK : 0);
        imap.put(KeyStroke.getKeyStroke(key, modifiers), name);
    }

    /**
     * I͈͂̍Ō̃Zɒu鑾gB
     */
    private static final class CellCursor {

        private JTable table;

        /**
         * CellCursor̐B
         * @param table
         */
        CellCursor(JTable table) {
            this.table = table;
        }

        /**
         * ʒu̎擾B
         * @return ʒu
         */
        int getColumnPosition() {
            int[] a = table.getSelectedColumns();
            if (a == null || a.length == 0) {
                return -1;
            }
            ListSelectionModel csm = table.getColumnModel().getSelectionModel();
            return (a[0] == csm.getAnchorSelectionIndex()) ? a[a.length - 1] : a[0];
        }

        /**
         * sʒu̎擾B
         * @return sʒu
         */
        int getRowPosition() {
            int[] a = table.getSelectedRows();
            if (a == null || a.length == 0) {
                return -1;
            }
            ListSelectionModel rsm = table.getSelectionModel();
            return (a[0] == rsm.getAnchorSelectionIndex()) ? a[a.length - 1] : a[0];
        }

    }

    /**
     * ADJUST_COLUMN_SIZE̊蓖āB
     */
    private void bindAdjustColumnSize() {
        String key = ADJUST_COLUMN_SIZE;
        amap.put(key, new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                table.adjustColumnWidth();
            }

        });
        imap.put(getKeyStroke(KeyEvent.VK_SLASH, true), key);
    }

    /**
     * COPY_CELL_WITH_ESCAPE̊蓖āB
     */
    private void bindCopyCellWithEscape() {
        String key = COPY_CELL_WITH_ESCAPE;
        amap.put(key, new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                copyValueAtSelectedCells();
            }

        });
        int mask = InputEvent.CTRL_DOWN_MASK | InputEvent.ALT_DOWN_MASK;
        imap.put(KeyStroke.getKeyStroke(KeyEvent.VK_C, mask), key);
    }

    /**
     * COPY_COLUMN_NAMES̊蓖āB
     */
    private void bindCopyColumnNames() {
        String key = COPY_COLUMN_NAMES;
        amap.put(key, new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                copyColumnNames();
            }

        });
    }

    /**
     * PASTE_TO_SELECTED_CELLS̊蓖āB
     */
    private void bindPasteToSelectedCells() {
        String key = PASTE_TO_SELECTED_CELLS;
        amap.put(key, new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                pasteValueToSelectedCells();
            }

        });
        imap.put(getKeyStroke(KeyEvent.VK_V, true), key);
    }

    /**
     * SET_NULL_TO_SELECTED_CELLS̊蓖āB
     */
    private void bindSetNullToSelectedCells() {
        String key = SET_NULL_TO_SELECTED_CELLS;
        amap.put(key, new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                resetValueAtSelectedCells();
            }

        });
        imap.put(getKeyStroke(KeyEvent.VK_DELETE, false), key);
        imap.put(getKeyStroke(KeyEvent.VK_R, true), key);
    }

    /**
     * ADD_NEW_ROW̊蓖āB
     */
    private void bindAddNewRow() {
        String key = ADD_NEW_ROW;
        amap.put(key, new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                addNewRow();
            }

        });
        int mask = InputEvent.CTRL_DOWN_MASK | InputEvent.SHIFT_DOWN_MASK;
        imap.put(KeyStroke.getKeyStroke(KeyEvent.VK_SEMICOLON, mask), key);
    }

    /**
     * ADD_DUPLICATE_ROW̊蓖āB
     */
    private void bindAddDuplicateRow() {
        String key = ADD_DUPLICATE_ROW;
        amap.put(key, new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                addDuplicateRows();
            }

        });
    }

    /**
     * INSERT_FROM_CLIPBOARD̊蓖āB
     */
    private void bindInsertFromClipboard() {
        String key = INSERT_FROM_CLIPBOARD;
        amap.put(key, new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                insertFromClipboard();
            }

        });
    }

    /**
     * INSERT_ADDED_ROWS̊蓖āB
     */
    private void bindInsertAddedRows() {
        String key = INSERT_ADDED_ROWS;
        amap.put(key, new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                insertAddedRows();
            }

        });
    }

    /**
     * DELETE_SELECTED_ROWS̊蓖āB
     */
    private void bindDeleteSelectedRows() {
        String key = DELETE_SELECTED_ROWS;
        amap.put(key, new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                deleteSelectedRows();
            }

        });
        int mask = InputEvent.CTRL_DOWN_MASK | InputEvent.SHIFT_DOWN_MASK;
        imap.put(KeyStroke.getKeyStroke(KeyEvent.VK_MINUS, mask), key);
    }

    /**
     * KeyStroke̎擾B
     * @param key L[
     * @param withControlKey Rg[L[gpꍇ <code>true</code>
     * @return KeyStroke
     */
    private static KeyStroke getKeyStroke(int key, boolean withControlKey) {
        return KeyStroke.getKeyStroke(key, (withControlKey ? InputEvent.CTRL_DOWN_MASK : 0));
    }

    /**
     * G[̏B
     * @param th O
     */
    void onError(Throwable th) {
        log.error("", th);
        DialogMessage.alert(table, th.getMessage());
    }

    /**
     * IZ̒lNbv{[hɃRs[B
     */
    void copyValueAtSelectedCells() {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        Exporter exporter = new SimpleExporter(bos, "\t");
        int[] selectedColumns = table.getSelectedColumns();
        int[] selectedRows = table.getSelectedRows();
        int columnCount = selectedColumns.length;
        try {
            for (int y = 0; y < selectedRows.length; y++) {
                int rowIndex = selectedRows[y];
                Object[] values = new Object[columnCount];
                for (int x = 0; x < columnCount; x++) {
                    int columnIndex = selectedColumns[x];
                    Object value = table.getValueAt(rowIndex, columnIndex);
                    if (value == null) {
                        value = "";
                    }
                    values[x] = value;
                }
                exporter.addRow(values);
            }
            String contents = bos.toString().replaceAll("\\r", "");
            if (log.isTraceEnabled()) {
                Writer buffer = new StringWriter();
                PrintWriter out = new PrintWriter(buffer);
                out.println("to clipboard:");
                out.print(contents);
                log.trace(buffer);
            }
            Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
            StringSelection ss = new StringSelection(contents);
            clipboard.setContents(ss, ss);
        } catch (IOException ex) {
            onError(ex);
        }
    }

    /**
     * 񖼂Nbv{[hɃRs[B
     */
    void copyColumnNames() {
        TableColumnModel columnModel = table.getTableHeader().getColumnModel();
        int columnCount = columnModel.getColumnCount();
        if (columnCount < 1) {
            return;
        }
        StringBuffer buffer = new StringBuffer();
        for (int i = 0; i < columnCount; i++) {
            if (i != 0) {
                buffer.append('\t');
            }
            buffer.append(columnModel.getColumn(i).getHeaderValue());
        }
        if (log.isTraceEnabled()) {
            log.trace("to clipboard:" + buffer);
        }
        Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
        StringSelection ss = new StringSelection(buffer.toString());
        clipboard.setContents(ss, ss);
    }

    /**
     * IZɃNbv{[h̒l\tB
     */
    void pasteValueToSelectedCells() {
        Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
        Transferable t = clipboard.getContents(null);
        try {
            String contents = (String)t.getTransferData(DataFlavor.stringFlavor);
            paste(table, contents);
        } catch (UnsupportedFlavorException ex) {
            onError(ex);
        } catch (IOException ex) {
            onError(ex);
        } catch (SQLException ex) {
            onError(ex);
        }
    }

    /**
     * \tsB
     * @param table e[u
     * @param contents \te
     * @throws IOException o̓G[ꍇ
     * @throws SQLException SQL֘AG[ꍇ
     */
    private static void paste(JTable table, String contents) throws IOException, SQLException {
        ResultSetTableModel model = (ResultSetTableModel)table.getModel();
        Importer importer = new TabSeparatedTextImporter(contents);
        try {
            int[] selectedColumns = table.getSelectedColumns();
            int[] selectedRows = table.getSelectedRows();
            int columnCount = selectedColumns.length;
            for (int y = 0; y < selectedRows.length; y++) {
                int rowIndex = selectedRows[y];
                boolean isInsertRow = model.isInsertRow(rowIndex);
                Object[] values = importer.nextRow();
                int limit = Math.min(columnCount, values.length);
                for (int x = 0; x < limit; x++) {
                    int columnIndex = selectedColumns[x];
                    Object oldValue = table.getValueAt(rowIndex, columnIndex);
                    Object newValue = values[x];
                    if (!isInsertRow) {
                        model.updateValueChanged(table, rowIndex, columnIndex, oldValue, newValue);
                    }
                    table.setValueAt(newValue, rowIndex, columnIndex);
                }
            }
        } finally {
            importer.close();
        }
    }

    /**
     * IZ̒lZbg(NullZbg)B
     */
    void resetValueAtSelectedCells() {
        try {
            ResultSetTableModel model = (ResultSetTableModel)table.getModel();
            int[] selectedColumns = table.getSelectedColumns();
            int[] selectedRows = table.getSelectedRows();
            int columnCount = selectedColumns.length;
            for (int y = 0; y < selectedRows.length; y++) {
                int rowIndex = selectedRows[y];
                for (int x = 0; x < columnCount; x++) {
                    int columnIndex = selectedColumns[x];
                    Object oldValue = table.getValueAt(rowIndex, columnIndex);
                    model.updateValueChanged(table, rowIndex, columnIndex, oldValue, null);
                    table.setValueAt(null, rowIndex, columnIndex);
                }
            }
        } catch (SQLException ex) {
            onError(ex);
        }
    }

    /**
     * sVKǉB
     */
    void addNewRow() {
        ResultSetTableModel model = (ResultSetTableModel)table.getModel();
        int rowCount = model.getRowCount();
        model.addRowAsInsert(new Object[model.getColumnCount()]);
        table.changeSelection(rowCount, 1, false, false);
    }

    /**
     * IꂽsRs[ĒǉB
     */
    void addDuplicateRows() {
        ResultSetTableModel model = (ResultSetTableModel)table.getModel();
        int[] selectedRows = table.getSelectedRows();
        for (int i = 0; i < selectedRows.length; i++) {
            List row = (List)model.getDataVector().get(selectedRows[i]);
            model.addRowAsInsert(row.toArray());
            table.changeSelection(model.getRowCount(), 1, false, false);
        }
    }

    /**
     * Nbv{[h̒l}B
     */
    void insertFromClipboard() {
        Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
        try {
            Transferable t = clipboard.getContents(null);
            Object value = t.getTransferData(DataFlavor.stringFlavor);
            ResultSetTableModel model = (ResultSetTableModel)table.getModel();
            Importer importer = new TabSeparatedTextImporter(String.valueOf(value));
            try {
                List errorList = new ArrayList();
                for (int rowIndex = table.getRowCount();; rowIndex++) {
                    Object[] row = importer.nextRow();
                    if (row.length == 0) {
                        break;
                    }
                    model.addRowAsInsert(row);
                    try {
                        model.updateRowInserted(table, rowIndex);
                    } catch (SQLException ex) {
                        log.warn("", ex);
                        errorList.add(ex);
                    }
                }
                if (!errorList.isEmpty()) {
                    String message = errorList.size() + " error(s) occurred";
                    DialogMessage.alert(table, message);
                }
            } finally {
                importer.close();
            }
        } catch (UnsupportedFlavorException ex) {
            onError(ex);
        } catch (IOException ex) {
            onError(ex);
        } catch (RuntimeException ex) {
            onError(ex);
        }
    }

    /**
     * ǉs̑}m肷B
     */
    void insertAddedRows() {
        try {
            ResultSetTableModel model = (ResultSetTableModel)table.getModel();
            int[] selectedRows = table.getSelectedRows();
            for (int i = 0; i < selectedRows.length; i++) {
                model.updateRowInserted(table, selectedRows[i]);
            }
        } catch (SQLException ex) {
            onError(ex);
        } catch (RuntimeException ex) {
            onError(ex);
        }
    }

    /**
     * Iꂽs폜B
     */
    void deleteSelectedRows() {
        ResultSetTableModel model = (ResultSetTableModel)table.getModel();
        int[] selectedRows = table.getSelectedRows();
        final int selectedRowCount = selectedRows.length;
        if (selectedRowCount < 1) {
            throw new IllegalStateException("Why called ResultSetTableAction.deleteSelectedRows() ?");
        }
        /*
         * (#236̑Ή)
         * ォ珇ɍ폜ĂAтɈsl߂B
         * ̂߃\[gĂKvB
         */
        Arrays.sort(selectedRows);
        int deletedCount = 0;
        for (int i = 0; i < selectedRowCount; i++) {
            final int selectedRow = selectedRows[i] - deletedCount;
            if (!model.isInsertRow(selectedRow)) {
                try {
                    model.updateRowDeleted(table, new int[]{selectedRow});
                } catch (SQLException ex) {
                    onError(ex);
                }
            }
            model.removeRow(selectedRow);
            ++deletedCount;
        }
    }

}
